# @copyright (c) 2002-2013 Acronis International GmbH. All rights reserved.

from string import Template
from collections import namedtuple
import colorsys
import re
import os


RGB = namedtuple('RGB', ('r', 'g', 'b'))


def color2rgb(color):
    if isinstance(color, RGB):
        return color
    if isinstance(color, str):
        color = int(color, 16)
    return RGB((color >> 16) & 255, (color >> 8) & 255, color & 255)


def rgb2css(rgb):
    return '#{:02X}{:02X}{:02X}'.format(*map(int, rgb))


def rgb_to_hls(rgb):
    return colorsys.rgb_to_hls(rgb.r / 255, rgb.g / 255, rgb.b / 255)


def hls_to_rgb(h, l, s):
    return RGB(*(c * 255 for c in colorsys.hls_to_rgb(h, l, s)))


def rgb2lab(rgb):
    def transform_rgb(x):
        x /= 255
        x = ((x + 0.055) / 1.055) ** 2.4 if x > 0.04045 else x / 12.92
        return x * 100

    def transform_xyz(x):
        return x ** 0.3333333333333333 if x > 0.008856 else (7.787 * x) + (16 / 116)

    rgb = RGB(transform_rgb(rgb.r), transform_rgb(rgb.g), transform_rgb(rgb.b))

    xyz = [
        transform_xyz(round(rgb.r * 0.4124 + rgb.g * 0.3576 + rgb.b * 0.1805, 4) / 95.047),
        transform_xyz(round(rgb.r * 0.2126 + rgb.g * 0.7152 + rgb.b * 0.0722, 4) / 100.0),
        transform_xyz(round(rgb.r * 0.0193 + rgb.g * 0.1192 + rgb.b * 0.9505, 4) / 108.883)
    ]

    return [
        round((116 * xyz[1]) - 16, 4),
        round(500 * (xyz[0] - xyz[1]), 4),
        round(200 * (xyz[1] - xyz[2]))
    ]


def color2css(color):
    return rgb2css(color2rgb(color))


def is_dark(rgb):
    return rgb2lab(rgb)[0] < 70


def add_black(rgb, amount):
    hue, saturation, value = colorsys.rgb_to_hsv(*rgb)
    return colorsys.hsv_to_rgb(hue, saturation, max(value * (1 - amount), 1))


def mixin(color1, color2, amount=1):
    color1 = color2rgb(color1)
    color2 = color2rgb(color2)

    return RGB(((color1.r + (color2.r * amount)) // 2),
               ((color1.r + (color2.r * amount)) // 2),
               ((color1.b + (color2.b * amount)) // 2))


def generate_palette(base_color, rules):
    """
        generate palette according to specified rules
        rules format is map {"name": "value"}
        where value can be on of following variations:
        int: added black color with specified amount
        tuple(color, amount): mixin base color with color according to specified amount
        any other type is used as is.
    """
    base_color = color2rgb(base_color)

    def transform(value):
        if isinstance(value, (float, int)):
            return rgb2css(add_black(base_color, value))
        if isinstance(value, (tuple, list)):
            return rgb2css(mixin(base_color, *value))
        return value

    index = is_dark(base_color)
    return {k: transform(v[index]) for k, v in rules.items()}


RE_BLOCK = re.compile(r'([^\{]+)\{\s*([^\}]+)\}')
RE_IMPORT = re.compile(r'@import\s+\'(.*?)\';?')


def extract_css_color(style): 
    m = re.search('color:\s*(#[a-zA-Z0-9]+)', style)
    if m and m.group(1):
        return m.group(1)
    
    raise ValueError('CSS color not found: ' + style)


def extract_css(style, color, replacement):
    return ';'.join([l.replace(color, replacement) for l in style.split(';') if color in l])


def extract_icon(style, scheme, variable='${{{0}}}'):
    style = ';'.join([l.replace(scheme, '-' + variable.format('acronisIconScheme')) for l in style.split(';') if scheme in l])
    style = style.replace('images/', '/resources/images/')
    return style


def generate_stylesheet(template, palette):
    template = Template(template)
    return template.safe_substitute(**palette)


def generate_template(source, target, palette, variable='${{{}}}'):
    history = []

    def lookup_in_history(regexp):
        for item in history:
            if regexp.search(item):
                return True
        return False

    with open(source, 'r') as stylesheet:
        with open(target, 'w') as template:
            for line in stylesheet:
                for name, value in palette.items():
                    if isinstance(value, list):
                        value = '|'.join(value)

                    if isinstance(value, dict) and lookup_in_history(re.compile(r'{}\.*\{{'.format(value['cls']), flags=re.IGNORECASE)):
                        # replace class color
                        line = re.sub(r'({0})\s*;'.format(value), variable.format(name) + r';', line, flags=re.IGNORECASE)
                    elif name.endswith('_color'):
                        # replace font colors
                        line = re.sub(r'(\s+)color:(\s+)({0})\s*;'.format(value), r'\1color:\2' + variable.format(name) + r';', line, flags=re.IGNORECASE)
                    elif name.endswith('_border'):
                        # replace border colors
                        line = re.sub(r'(.*)border(.*)({0})\s*;'.format(value), r'\1border\2' + variable.format(name) + r';', line, flags=re.IGNORECASE)
                    else:
                        # replace all others
                        line = re.sub(r'({0})\s*;'.format(value), variable.format(name) + r';', line, flags=re.IGNORECASE)

                if '}' in line:
                    history.clear()
                else:
                    history.append(line)
                template.write(line)


def generate_template_from_css(css_file_name, template_file_name, color_blocks, special_rules=None, variable='${{{0}}}'):
    base_colors = {}
    special_rules = special_rules or {}

    with open(css_file_name, 'r') as stylesheet:
        css_content = stylesheet.read()
        m = re.search(RE_IMPORT, css_content)
        
        while m:
            with open(os.path.join(os.path.dirname(css_file_name), m.group(1)), 'r') as f:
                imported_content = f.read()
                css_content = css_content[:m.start()] + imported_content + '\n' + css_content[m.end():]
            m = re.search(RE_IMPORT, css_content)
       
        with open(template_file_name, 'w+') as template:
            blocks = re.findall(RE_BLOCK, css_content)
            
            for name, style in blocks:
                # load base colors at the beginning of CSS
                if name in color_blocks:
                    base_colors[extract_css_color(style)] = variable.format(name[1:])
                    continue
            
                for color in base_colors:
                    if color in style:
                        template.write("%s {%s}\n" % (name, extract_css(style, color, base_colors[color])))

                if '-dark.png' in style or '-dark.gif' in style:
                    template.write("%s {%s}\n" % (name, extract_icon(style, '-dark', variable)))
                if '-light.png' in style or '-light.gif' in style:
                    template.write("%s {%s}\n" % (name, extract_icon(style, '-light', variable)))

            template.write('\n')
            for name in special_rules:
                style = special_rules[name]
                template.write("%s {%s: %s}\n" % (name, style[0], variable.format(style[1])))


def resolve_colors(*scss_files, **kwargs):
    regexp = re.compile(r'\s*(\$[\d\w\-]+)\s*:\s*(#[a-fA-F0-9]{6}|\$[\d\w\-]+|\w+).*')
    variables = {}
    palette = kwargs['pallete']

    def resolve_color(color):
        variable = variables.get(color, None)
        if variable is None:
            return color
        if variable.startswith('$'):
            return resolve_color(variable)
        return variable

    for scss_file_path in scss_files:
        with open(scss_file_path, 'r') as file:
            for line in file:
                m = regexp.search(line)
                if m:
                    variables[m.group(1).lower()] = m.group(2).lower()

    resolved_palette = {}
    for name, value in palette.items():
        if isinstance(value, list):
            resolved_palette[name] = [resolve_color(color) for color in value]
        elif isinstance(value, dict):
            value_copy = value.copy()
            value_copy.update(color=resolve_color(value['color']))
            resolved_palette[name] = value_copy
        else:
            resolved_palette[name] = resolve_color(value)
    return resolved_palette


def dump_palette(palette):
    result = '{\n'
    for key, value in palette.items():
        if isinstance(value, dict):
            value = value['color']
        elif isinstance(value, list):
            value = value[0]

        result += '    {0}: {1}, \n'.format(key, value)
    result += '}'
    return result
