# Run as 'python.exe -m acrobind.staging.test_retention_by_size' from enterprise/pybind folder

import acrobind.staging.retention as retention
from acrobind.staging import *

import datetime
import unittest

# Ordering of generated slices is from older to newer
class TestRetentions(unittest.TestCase):

    def test_single_slice_cant_be_retained_before_limit(self):
        archive = {'size': 2100}
        slices = [{'id': 1110, 'size': 1000, 'kind': SLICE_KIND_FULL}]
        limit = 2000
        abandoned_slices = retention.calculate_slices_abandoned_by_archive_size(archive, slices, limit)
        #print('abandoned slices: ', abandoned_slices)
        self.assertTrue(len(abandoned_slices) == 0)


    def test_single_slice_cant_be_retained_above_limit(self):
        archive = {'size': 2100}
        slices = [{'id': 1110, 'size': 2100, 'kind': SLICE_KIND_FULL}]
        limit = 2000
        abandoned_slices = retention.calculate_slices_abandoned_by_archive_size(archive, slices, limit)
        #print('abandoned slices: ', abandoned_slices)
        self.assertTrue(len(abandoned_slices) == 0)


    def test_two_slices_cant_be_retained_when_fit_exactly_to_limit(self):
        archive = {'size': 2100}
        slices = [{'id': 1110, 'size': 1000, 'kind': SLICE_KIND_FULL}]
        slices += [{'id': i, 'size': 1000, 'kind': SLICE_KIND_INCREMENTAL} for i in range(1, 3)]
        limit = 2000
        abandoned_slices = retention.calculate_slices_abandoned_by_archive_size(archive, slices, limit)
        #print('abandoned slices: ', abandoned_slices)
        self.assertTrue(len(abandoned_slices) == 1)


    def test_single_slice_should_left_after_retention_of_slice_chain(self):
        archive = {'size': 5000}
        slices = [{'id': 1110, 'size': 1000, 'kind': SLICE_KIND_FULL}]
        slices += [{'id': i, 'size': i*100, 'kind': SLICE_KIND_INCREMENTAL} for i in range(1, 10)]
        slice_id_should_left = 1111
        slices += [{'id': slice_id_should_left, 'size': 1000, 'kind': SLICE_KIND_FULL}]
        limit = 100
        abandoned_slices = retention.calculate_slices_abandoned_by_archive_size(archive, slices, limit)
        print('abandoned slices: ', abandoned_slices)
        for slice in abandoned_slices:
            self.assertTrue(slice['id'] != slice_id_should_left)
        self.assertTrue(len(slices) - len(abandoned_slices) == 1)


    def test_all_slices_should_left_until_limit(self):
        archive = {'size': 2000}
        slices = [{'id': 1111, 'size': 1000, 'kind': SLICE_KIND_FULL}]
        slices += [{'id': i, 'size': i*100, 'kind': SLICE_KIND_INCREMENTAL} for i in range(1, 10)]
        limit = 2000
        abandoned_slices = retention.calculate_slices_abandoned_by_archive_size(archive, slices, limit)
        #print('abandoned slices: ', abandoned_slices)
        self.assertTrue(len(abandoned_slices) == 0)


    def test_last_slice_should_left_after_limit(self):
        archive = {'size': 8000}
        slices = [{'id': 1111, 'size': 1000, 'kind': SLICE_KIND_FULL}]
        slices += [{'id': i, 'size': i*100, 'kind': SLICE_KIND_INCREMENTAL} for i in range(1, 10)]
        slice_id_should_left = 1112
        slices += [{'id': slice_id_should_left, 'size': 1000, 'kind': SLICE_KIND_INCREMENTAL}]
        limit = 1000
        abandoned_slices = retention.calculate_slices_abandoned_by_archive_size(archive, slices, limit)
        #print('abandoned slices: ', abandoned_slices)
        for slice in abandoned_slices:
            self.assertTrue(slice['id'] != slice_id_should_left)
        self.assertTrue(len(slices) - len(abandoned_slices) == 1)


    def test_dont_retain_full_slice_if_diff_slice_present_in_chain(self):
        archive = {'size': 5000}
        slices = [{'id': 1, 'size': 1000, 'kind': SLICE_KIND_FULL}]

        for i in range(1,3):
            slices += [{'id': i*2, 'size': 1000, 'kind': SLICE_KIND_INCREMENTAL}]
            slices += [{'id': i*2+1, 'size': 1000, 'kind': SLICE_KIND_DIFFERENTIAL}]

        full_slice_id_should_left = 1112
        slices += [{'id': full_slice_id_should_left, 'size': 1000, 'kind': SLICE_KIND_FULL}]

        for i in range(3,5):
            slices += [{'id': i*2, 'size': 1000, 'kind': SLICE_KIND_INCREMENTAL}]
            slices += [{'id': i*2+1, 'size': 1000, 'kind': SLICE_KIND_DIFFERENTIAL}]

        diff_slice_id_should_left = 1113
        slices += [{'id': diff_slice_id_should_left, 'size': 1000, 'kind': SLICE_KIND_DIFFERENTIAL}]

        limit = 1000
        abandoned_slices = retention.calculate_slices_abandoned_by_archive_size(archive, slices, limit)
        #print('abandoned slices: ', abandoned_slices)
        slices_should_left = [full_slice_id_should_left, diff_slice_id_should_left]
        for slice in abandoned_slices:
            self.assertTrue(slice['id'] not in slices_should_left)
        self.assertTrue(len(slices) - len(abandoned_slices) == len(slices_should_left))


if __name__ == '__main__':
    unittest.main()
