import http.client
import http.cookiejar
import urllib.parse
import json
import ssl


class HttpUnexpectedResponseStatus(Exception):
    def __init__(self, status, data):
        Exception.__init__(self)
        try:
            self._data = json.loads(data.decode('utf-8'))
        except ValueError:
            self._data = {}
        self._status = status

    def __str__(self):
        return "An unexpected response status '{} ({})'.".format(self._status, http.client.responses.get(self._status, 'Unknown'))


class HttpApiClient:
    def __init__(self, *, url, port=None, use_ssl=None, use_ssl_unverified_context=False, debug=None):
        self._connected = False
        if url.startswith('https://'):
            use_ssl = True
            url = url[8:]
        if use_ssl:
            ssl_context = None
            if use_ssl_unverified_context:
                ssl_context = ssl._create_unverified_context()
            self._connection = http.client.HTTPSConnection(url, port=port, context=ssl_context)
        else:
            if url.startswith('http://'):
                url = url[7:]
            self._connection = http.client.HTTPConnection(url, port=port)
        if use_ssl:
            self._req_url = 'https://' + url
        else:
            self._req_url = 'http://' + url
        self._connection.connect()
        self._connected = True
        self._jar = http.cookiejar.CookieJar()
        self._debug = debug

    def close(self):
        if self._connected:
            self._connected = False
            self._connection.close()

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        self.close()

    def __del__(self):
        self.close()

    def debug(self, text):
        if self._debug:
            print('DEBUG: {}'.format(text))

    def get(self, url):
        response = self._perform_request(url, 'GET')
        return self._handle_response(response, self._parse_json)

    def get_response(self, url):
        self.debug("GET {}".format(url))
        headers = self._build_headers()
        self._connection.request('GET', url, headers=headers)
        return self._connection.getresponse()

    def post(self, url, data=None):
        response = self._perform_request(url, 'POST', data if data else {}, 'application/json; charset=utf-8')
        return self._handle_response(response, self._parse_json)

    def delete(self, url, data=None):
        response = self._perform_request(url, 'DELETE', data if data else {}, 'application/json; charset=utf-8')
        return self._handle_response(response, self._parse_json)

    def put(self, url, data=None):
        response = self._perform_request(url, 'PUT', data if data else {}, 'application/json; charset=utf-8')
        return self._handle_response(response, self._parse_json)

    def _build_headers(self, content_type=None):
        headers = {}
        mock_request = urllib.request.Request(self._req_url)
        self._jar.add_cookie_header(mock_request)
        cookie_header = mock_request.get_header('Cookie')
        if cookie_header is not None:
            headers['Cookie'] = cookie_header
        if content_type is not None:
            headers['Content-type'] = content_type
        return headers

    def _perform_request(self, url, method, data=None, content_type=None):
        self.debug("{} {}".format(method, url))
        headers = self._build_headers(content_type)

        body = None
        if data is not None:
            if isinstance(data, list):
                raw_data = data
            else:
                raw_data = self._to_dict(data)
            self.debug('DATA:\n{0}'.format(json.dumps(raw_data, indent=2)))
            body = json.dumps(raw_data)

        while True:
            self._connection.request(method, url, body=body, headers=headers)
            response = self._connection.getresponse()
            if response.status not in (http.client.MOVED_PERMANENTLY, http.client.FOUND):
                return response
            response.read()
            url = urllib.parse.urljoin(url, response.getheader('location', ''))

    def _handle_response(self, response, result_handler):
        data = response.read()
        if response.status in (http.client.OK, http.client.CREATED, http.client.ACCEPTED):
            mock_request = urllib.request.Request(self._req_url)
            self._jar.extract_cookies(response, mock_request)
            return result_handler(data) if data else {}
        elif response.status in (http.client.NO_CONTENT, http.client.SEE_OTHER):
            return {}
        elif response.status == http.client.NOT_MODIFIED:
            return int(response.getheader('retry-after'))
        else:
            try:
                logged_data = json.dumps(json.loads(data), indent=2)
            except Exception:
                logged_data = data
            self.debug('RESPONSE:\n{0}'.format(logged_data))
            raise HttpUnexpectedResponseStatus(response.status, data)

    def _to_dict(self, obj):
        if isinstance(obj, dict):
            return obj

        data = {}
        for key, value in obj.__dict__.items():
            if key.startswith('__'):
                continue
            name = key.lstrip('_')
            try:
                if isinstance(value, list):
                    data[name] = [self._to_dict(iter) for iter in value]
                else:
                    data[name] = self._to_dict(value)
            except AttributeError:
                data[name] = value
        return data

    def _parse_json(self, data):
        result = json.loads(data.decode('utf-8'))
        self.debug('RESULT:\n{0}'.format(json.dumps(result, indent=2)))
        return result
