import acrort
import re
import sys

from . registry_utils import registry_read_string


def _get_ams_connection_string(db_name):
    key_path = r'SOFTWARE\{}\AMS\ConnectionStrings'.format(acrort.common.BRAND_NAME)
    string = registry_read_string(key_path, db_name)
    if string:
        return string
    if acrort.common.WOW6432:
        string = registry_read_string(key_path, db_name, hive=acrort.registry.HIVE_HKLM32)
    if string:
        return string
    return None


def _is_trusted_connection(connection_params):
    return ("Integrated Security" in connection_params)


def _format_windows_authentification(connection_params):
    return "Trusted_Connection={trusted}".format(trusted=connection_params["Integrated Security"])


def _format_sql_authentification(connection_params):
    return "UID={user};PWD={password}".format(user=connection_params["User Id"], password=connection_params["Password"])


def _adapt_ms_sql_connection_string(connection_string):
    pattern = re.compile("([^;]+)=([^;]+);{0,1}")
    connection_params = dict(pattern.findall(connection_string))
    base_connection_string = "DRIVER={{SQL Server}};Server={instance};Database={db_name};".format(
        instance=connection_params["Data Source"], db_name=connection_params["Initial Catalog"])
    if _is_trusted_connection(connection_params):
        return base_connection_string + _format_windows_authentification(connection_params)
    else:
        return base_connection_string + _format_sql_authentification(connection_params)


def connect_to_database(db_name):
    try:
        conn_string = _get_ams_connection_string(db_name)
    except acrort.Exception:
        details = acrort.common.get_current_exception_as_error()
        acrort.common.make_logic_error_with_suberror("Cannot read connection string for database '{}' from registry.".format(db_name), details).throw()

    if conn_string is None:
        acrort.common.make_logic_error("Connection string for database '{}' not found in registry.".format(db_name)).throw()

    if sys.platform == 'win32':
        adapted_conn_string = _adapt_ms_sql_connection_string(conn_string)
        import pyodbc
        return pyodbc.connect(adapted_conn_string)
    else:
        import psycopg2
        return psycopg2.connect(conn_string)


def get_ams_dml_connection_string():
    conn_string = None
    key = r'SOFTWARE\{}\AMS\Settings'.format(acrort.common.BRAND_NAME)
    value = 'AmsDmlDbProtocol'
    try:
        conn_string = registry_read_string(key, value)
    except acrort.Exception:
        details = acrort.common.get_current_exception_as_error()
        acrort.common.make_logic_error_with_suberror(
            "Cannot read DML connection string from registry, service: 'ams', key: '{}', value: '{}'.".format(key, value), details).throw()
    if conn_string is None:
        acrort.common.make_logic_error("DML connection string not found in registry, service: 'ams', key: '{}', value: '{}'.".format(key, value)).throw()
    return string
