# @copyright (c) 2002-2013 Acronis International GmbH. All rights reserved.

from acronis.lib.unrepr import unrepr
from configparser import ConfigParser, ExtendedInterpolation
from collections import defaultdict
from io import StringIO
import fnmatch
import os
import re
import shutil


class _Preprocessor:
    def __init__(self):
        self.include_expression = re.compile(r'#include\s+"([\w\d/\.\*]+)"', flags=re.IGNORECASE)
        self.workdir = None
        self.output = None
        self.encoding = None

    def include(self, source):
        shutil.copyfileobj(_Preprocessor.compile(source, self.encoding), self.output)

    def process_include(self, include):
        if '*' in include:
            dirname = os.path.join(os.path.dirname(include))
            for filename in fnmatch.filter(os.listdir(dirname), os.path.basename(include)):
                self.include(os.path.join(dirname, filename))
        else:
            self.include(include)

    def process_line(self, line):
        tokens = self.include_expression.match(line)
        if tokens:
            self.process_include(os.path.join(self.workdir, tokens.groups()[0]))
        else:
            self.output.write(line)

    @staticmethod
    def compile(filename, encoding):
        context = _Preprocessor()
        context.encoding = encoding
        context.workdir = os.path.dirname(filename)
        context.output = StringIO()
        with open(filename, 'r', encoding=context.encoding) as source_stream:
            for line in source_stream.readlines():
                context.process_line(line)
        context.output.seek(0)
        return context.output


class Parser(ConfigParser):
    """
        Sub-class of ConfigParser that keeps the case of options and that
        raises an exception if the file cannot be read.
    """
    _DEFAULT_INTERPOLATION = ExtendedInterpolation()

    def get(self, *args, raw=False, **kwargs):
        v = super().get(*args, raw=raw, **kwargs)
        return v.strip('"') if raw else v

    def read(self, filename, encoding=None):
        self._read(_Preprocessor.compile(filename, encoding), filename)

    def as_dict(self, raw=False, vars=None):
        """Convert an config to a dictionaries"""
        result = defaultdict(dict)
        for section in self.sections():
            for option in self.options(section):
                value = self.get(section, option, raw=raw, vars=vars)
                try:
                    value = value if raw else unrepr(value)
                except Exception as e:
                    msg = 'Config error in section: {0!r}, option: {1!r}, value {2!r}. Config values must be valid Python'.format(section, option, value)
                    raise ValueError(msg, e.__class__.__name__, e.args)
                result[section][option] = value
        return result

    def dict_from_file(self, file, vars=None):
        if hasattr(file, 'read'):
            self.read_file(file)
        else:
            self.read(file)
        return self.as_dict(vars=vars)


def load_from_file(file, vars=None):
    if file:
        return Parser(strict=False).dict_from_file(file, vars=vars)
    return defaultdict(dict)
