from . constants import *
from . slices import sort_by_creation_time
from calendar import monthrange
import datetime
import acrort

def relative_date(time, years=0, months=0):
    new_year = time.year + years + int(months / 12)
    new_month = time.month + (months % 12)
    if new_month > 12:
        new_month = new_month % 12
        new_year = new_year + 1
    max_day = monthrange(new_year, new_month)[1]
    new_day = min(time.day, max_day)

    result = time.replace(year=new_year, month=new_month, day=new_day)
    return result

def is_rule_for_all_backup_set(rule):
    if not rule.BackupSetIndex.is_data():
        return False
    return rule.BackupSetIndex.type == acrort.plain.NIL or rule.BackupSetIndex.ref == BACKUP_SET_ALL

def is_rule_for_one_backup_set(rule):
    if not rule.BackupSetIndex.is_data():
        return False
    return rule.BackupSetIndex.type != acrort.plain.NIL and rule.BackupSetIndex.ref != BACKUP_SET_ALL

def make_abandon_time_calculator(period_type, period_value):
    creation_time = lambda slice: slice['creation_time']
    period_types = {
        TIME_PERIOD_YEARS: (lambda x: relative_date(creation_time(x), years=period_value)),
        TIME_PERIOD_MONTHS: (lambda x: relative_date(creation_time(x), months=period_value)),
        TIME_PERIOD_WEEKS: (lambda x: creation_time(x) + datetime.timedelta(weeks=period_value)),
        TIME_PERIOD_DAYS: (lambda x: creation_time(x) + datetime.timedelta(days=period_value)),
        TIME_PERIOD_HOURS: (lambda x: creation_time(x) + datetime.timedelta(hours=period_value)),
        TIME_PERIOD_MINUTES: (lambda x: creation_time(x) + datetime.timedelta(minutes=period_value)),
        TIME_PERIOD_SECONDS: (lambda x: creation_time(x) + datetime.timedelta(seconds=period_value)),
        TIME_PERIOD_MILLISECONDS: (lambda x: creation_time(x) + datetime.timedelta(milliseconds=period_value))
    }
    return period_types[period_type]

def create_abandon_time_calculator(rule):
    period_value = rule.DeleteOlderBackupsTimePeriod.Value.ref
    period_type = rule.DeleteOlderBackupsTimePeriod.PeriodType.ref
    return make_abandon_time_calculator(period_type, period_value)


def check_slice_retention(slice, predecessor, abandon_time_calculators, backup_set_calculator, cleanup_time):
    backup_set = backup_set_calculator(slice, predecessor)
    if backup_set in abandon_time_calculators:
        abandon_time = abandon_time_calculators[backup_set]
        return abandon_time(slice) >= cleanup_time
    return True


def create_retention_predicate(rules, backup_set_calculator, cleanup_time):
    all_backup_set_rules = [rule for rule in rules if is_rule_for_all_backup_set(rule)]
    if len(all_backup_set_rules) > 0:
        adandon_time = create_abandon_time_calculator(all_backup_set_rules[BACKUP_SET_ALL])
        return (lambda slice, pred: adandon_time(slice) >= cleanup_time)
    backup_set_rules = [item for item in rules if is_rule_for_one_backup_set(item)]
    abandon_time_calculators = {rule.BackupSetIndex.ref: create_abandon_time_calculator(rule) for rule in backup_set_rules}
    return (lambda slice, predecessor: check_slice_retention(slice, predecessor, abandon_time_calculators, backup_set_calculator, cleanup_time))


# NEVER delete last slice, except setting 'Clean up archive > When there is insufficient space while backing up'
def preserve_last_slice(source_slices, affected_slices):
    if affected_slices and len(source_slices) == len(affected_slices):
        return affected_slices[:-1]
    
    return affected_slices


# We can't remove full slice if differential slice present in the live chain until next full slice
def preserve_full_slices(source_slices, abandoned_slices):
    # | F | D | D | F | D | F - initial chain (from older to newer)
    # | D | D | F | D | F     - full slice abandoned (incorrect way)

    # | F | D | D | F | D | F - initial chain (from older to newer)
    # | F | D | F | D | F     - 1. diff slice abandoned at first (correct way)
    # | F | F | D | F         - 2. diff slice abandoned at second (correct way)

    if not source_slices or not abandoned_slices:
        return []

    # Split live slices to chains (F, D, D)
    def extract_full_to_diff_slices_chain(slices):
        full_to_diff_slices_map = dict()
        current_full_slice_id = None
        for slice in slices:
            if slice['kind'] == SLICE_KIND_FULL:
                current_full_slice_id = slice['id']
            elif slice['kind'] == SLICE_KIND_DIFFERENTIAL:
                if current_full_slice_id:
                    v = full_to_diff_slices_map.setdefault(current_full_slice_id, set())
                    v.add(slice['id'])
        return full_to_diff_slices_map

    # Detect live slices chain closest to abandoned slices, then decide if we can remove full slice in abandoned ones
    skip_full_slice_ids = set()
    live_full_to_diff_slices_chain = extract_full_to_diff_slices_chain(source_slices)
    abandoned_full_to_diff_slices_chain = extract_full_to_diff_slices_chain(abandoned_slices)
    for full_slice_id, diff_slices in abandoned_full_to_diff_slices_chain.items():
        if live_full_to_diff_slices_chain.get(full_slice_id) != diff_slices:
            skip_full_slice_ids.add(full_slice_id)

    if not skip_full_slice_ids:
        return abandoned_slices # No differential backups in live slices chains, so remove all abandoned slices freely
    
    return [slice for slice in abandoned_slices if slice['id'] not in skip_full_slice_ids]


def select_slices_on_source(source_slice_selector):
    archive, source_slices = source_slice_selector()
    if not source_slices:
        return archive, []
    
    return archive, sort_by_creation_time(source_slices)


def select_slices_abandoned_by_time(source_slices, retention_predicate):
    if not source_slices:
        return []
    slices_and_predecessors = [(slice, pred) for slice, pred in zip(source_slices, [None] + source_slices[:-1])]
    abandoned_slices = [slice for slice, pred in slices_and_predecessors if not retention_predicate(slice, pred)]
    return preserve_full_slices(source_slices, abandoned_slices)


# Return all slices which exceed slice count limit
# - slices should be sorted from older to newer
# - special condition when we can't clean up 'full' slice if 'differential' slice should have been left in the archive
def calculate_slices_abandoned_by_slice_count(source_slices, upper_limit):
    if len(source_slices) <= upper_limit:
        return []

    return preserve_full_slices(source_slices, source_slices[:-upper_limit])


def select_slices_abandoned_by_number(source_slices, backup_set_calculator, rules):
    if not source_slices:
        return []

    all_backup_set_rules = [rule for rule in rules if is_rule_for_all_backup_set(rule)]
    if len(all_backup_set_rules) > 0:
        upper_limit = all_backup_set_rules[BACKUP_SET_ALL].BackupCountUpperLimit.ref
        return calculate_slices_abandoned_by_slice_count(source_slices, upper_limit)

    slices_and_predecessors = zip(source_slices, [None] + source_slices[:-1])
    slice_dict = {}
    for slice, predecessor in slices_and_predecessors:
        backup_set = backup_set_calculator(slice, predecessor)
        if not backup_set in slice_dict:
            slice_dict[backup_set] = [slice]
        else:
            slice_dict[backup_set] += [slice]

    abandoned = []
    backup_set_rules = [rule for rule in rules if is_rule_for_one_backup_set(rule)]
    backup_set_rules = {rule.BackupSetIndex.ref: rule.BackupCountUpperLimit.ref for rule in backup_set_rules}
    for backup_set, slices in slice_dict.items():
        if backup_set in backup_set_rules:
            upper_limit = backup_set_rules[backup_set]
            if len(slices) > upper_limit:
                abandoned += slices[:-upper_limit]
    
    return preserve_full_slices(source_slices, sort_by_creation_time(abandoned))


# Return all slices which exceed upper_limit_size
# - slices should be sorted from older to newer
# - there should be 2 or more slices
# - special condition when we can't clean up 'full' slice if 'differential' slice should be left in the archive
def calculate_slices_abandoned_by_archive_size(archive, slices, upper_limit_size):
    abandoned_slices = []
    if len(slices) < 2 or archive['size'] <= upper_limit_size:
        
        return abandoned_slices

    slices_size = 0
    slices_count = len(slices)
    for slice in reversed(slices):
        slices_size += slice['size']
        if slices_size >= upper_limit_size:
            abandoned_slices = slices[:slices_count - 1]
            break
        slices_count -= 1

    return preserve_full_slices(slices, abandoned_slices)


def select_slices_abandoned_by_archive_size(archive, source_slices, rules):
    if not source_slices:
        return []
    all_backup_set_rules = [rule for rule in rules if is_rule_for_all_backup_set(rule)]
    if len(all_backup_set_rules) > 0:
        return calculate_slices_abandoned_by_archive_size(
            archive, source_slices, all_backup_set_rules[BACKUP_SET_ALL].BackupUpperLimitSize.ref)

    raise ValueError('Retention by archive size and per backup set is not supported')
