import acrort
import datetime
import platform
from acrobind.protection.vaults import *
from urllib.parse import urlparse, parse_qs
from . command_ids import *
from . constants import *
from . backup_set import *
from . retention import *
from . replication import *
from . slices import *

_LOCATION_KIND_ONLINE = 10

def make_command_spec(command_id, argument, min_progress, max_progress):
    return {
        'CommandID': ('guid', command_id),
        'Argument': argument,
        'MinProgress': ('dword', min_progress),
        'MaxProgress': ('dword', max_progress)
    }


def make_workflow(commands):
    result = {'Commands': commands}
    return acrort.plain.Unit(result)


def request_for_make_shallow_copy(source, target, owner_id, create_pdsi_archive=False):
    argument = {
        'ArchiveName': source['archive_name'],
        'SourceLocation': {
            'Address': source['location_uri'],
            'Credentials': source['location_credentials']
        },
        'TargetLocation': {
            'Address': target['location_uri'],
            'Credentials': target['location_credentials']
        },
        'OwnerID': owner_id,
        'CreatePdsiArchive': create_pdsi_archive,
    }
    return (DMS_MAKE_SHALLOW_COPY_COMMAND_ID, argument)


def get_max_chain_length(slicing):
    return slicing.get('MaxValue', ('dword', 0))


def get_consolidate_backup_from_rules(rules):
    consolidate_backup_values = list({rule.ConsolidateBackup.ref for rule in rules})
    if len(consolidate_backup_values) != 1:
        raise ValueError(
            'ConsolidateBackup should have the same value in all rules but set of its values is: {}'.
                format(str(consolidate_backup_values)))
    return consolidate_backup_values[0]


def request_for_replicate_slices(source, slice_keys, target, options, owner_id, target_archive_is_pds=False, prohibit_location_creation=False):
    argument = {
        'SourceLocationUri': source['location_uri'],
        'SourceLocationLogon': source['location_credentials'],
        'SourceArchiveName': source['archive_name'],
        'SourceArchiveDisplayName': source['archive_display_name'],
        'SourceSliceKeys': acrort.plain.Unit(slice_keys),
        'TargetLocationUri': target['location_uri'],
        'TargetLocationLogon': target['location_credentials'],
        'TargetArchiveName': target['archive_name'],
        'TargetArchiveDisplayName': target['archive_display_name'],
        'ArchiveSlicing': source['archive_slicing'],
        'BackupOptions' : options.BackupOptions,
        'OwnerID': owner_id,
        'TaskExecutionWindow': target['task_execution_window'],
        'ProhibitLocationCreation': prohibit_location_creation,
        'TargetArchiveIsPDS': target_archive_is_pds,
    }
    return (DMS_REPLICATE_BACKUP_COMMAND_ID, argument)


def request_for_delete_slices(backup_plan, stage, slice_uris, consolidate_backup, owner_id):
    argument = {
        'Slices': slice_uris,
        'LocationCredentials': stage['location_credentials'],
        'ConsolidateBackup': consolidate_backup,
        'ArchiveName': stage['archive_name'],
        'OwnerID': owner_id,
    }
    if backup_plan['Options']['BackupOptions']['ArchiveProtection']['UsePassword'].ref:
        argument.update({'ArchiveCredentialsId': backup_plan['Options']['BackupOptions']['ArchiveProtection']['PasswordUrl']})
    return (DMS_DELETE_SLICES_COMMAND_ID, argument)

def request_for_delete_archive(stage, archive):
    argument = {
        'ArchivesUri': [archive],
        'LocationCredentials': stage['location_credentials'],
    }    
    return (DMS_DELETE_ARCHIVES_COMMAND_ID, argument)


def request_for_pause_cdp(plan):
    argument = {
        'PlanID': plan['ID'].ref,
    }
    return [(PAUSE_CDP_COMMAND_ID, argument)]

def request_for_resume_cdp(plan):
    argument = {
        'PlanID': plan['ID'].ref,
    }
    return [(RESUME_CDP_COMMAND_ID, argument)]

def is_cdp_plan(plan):
    opts = plan["Options"]["BackupOptions"]
    if "Cdp" not in opts:
        return False

    for t in opts["Cdp"].traits:
        if t[0] == 'is_null':
            return not t[1].ref

    return False


def calculate_just_cleanup_workflow(backup_plan, just_cleanup_rules, backup_set_calculator, cleanup_time,
    source_slice_selector, source_stage, retention_predicate_creator, owner_id):
    just_cleanup_by_time_rules = [item for item in just_cleanup_rules
        if item.BackupCountUpperLimit.ref == 0 and item.DeleteOlderBackupsTimePeriod.Value.ref > 0]
    just_cleanup_by_number_rules = [item for item in just_cleanup_rules
        if (item.BackupCountUpperLimit.ref > 0 and item.DeleteOlderBackupsTimePeriod.Value.ref == 0) or
           (item.BackupCountUpperLimit.ref == 0 and item.DeleteOlderBackupsTimePeriod.Value.ref == 0 and item.BackupUpperLimitSize.ref == 0 and item.BeforeBackup.ref) ]
    just_cleanup_by_archive_size_rules = [item for item in just_cleanup_rules
        if item.BackupUpperLimitSize.ref > 0 and item.DeleteOlderBackupsTimePeriod.Value.ref == 0]

    source_archive, source_slices = select_slices_on_source(source_slice_selector)
    if not source_archive:
        return []
    if just_cleanup_by_number_rules:
        item = just_cleanup_by_number_rules[0]
        if item.BackupCountUpperLimit.ref == 0 and item.DeleteOlderBackupsTimePeriod.Value.ref == 0 and item.BackupUpperLimitSize.ref == 0 and item.BeforeBackup.ref:
            return [request_for_delete_archive(
            source_stage,  source_archive['uri'])]

    slices = []
    if just_cleanup_by_time_rules:
        retention_predicate = retention_predicate_creator(
            just_cleanup_by_time_rules, backup_set_calculator, cleanup_time)
        slices = select_slices_abandoned_by_time(source_slices, retention_predicate)
    elif just_cleanup_by_number_rules:
        slices = select_slices_abandoned_by_number(
            source_slices, backup_set_calculator, just_cleanup_by_number_rules)
    elif just_cleanup_by_archive_size_rules:
        slices = select_slices_abandoned_by_archive_size(
            source_archive, source_slices, just_cleanup_by_archive_size_rules)
    if len(slices) == 0:
        return []

    slice_uris = [slice['uri'] for slice in slices]
    consolidate_backup = get_consolidate_backup_from_rules(just_cleanup_rules)
    return [request_for_delete_slices(backup_plan,
        source_stage, preserve_last_slice(source_slices, slice_uris), consolidate_backup, owner_id)]


def calculate_copy_on_appearance_workflow(copy_on_appearance_rules, backup_set_calculator,
    source_slice_selector, target_slice_selector, source_stage, target_stage, options, owner_id, target_archive_is_pds=False, prohibit_location_creation=False):
    replication_predicate = create_replication_predicate(copy_on_appearance_rules, backup_set_calculator)
    create_archive, slices = select_slices_to_replicate(
        source_slice_selector, target_slice_selector, replication_predicate)
    if len(slices) == 0:
        return []
    slice_keys = [slice['id'] for slice in slices]
    requests = []
    if create_archive:
        requests = [request_for_make_shallow_copy(source_stage, target_stage, owner_id)]
    return requests + [request_for_replicate_slices(source_stage, slice_keys, target_stage, options, owner_id, target_archive_is_pds, prohibit_location_creation)]


def calculate_move_on_cleanup_workflow(backup_plan, service, move_on_cleanup_rules, backup_set_calculator, cleanup_time,
    source_slice_selector, source_stage, target_stage, options, owner_id):
    move_on_cleanup_by_time_rules = [item for item in move_on_cleanup_rules
        if item.BackupCountUpperLimit.ref == 0 and item.DeleteOlderBackupsTimePeriod.Value.ref > 0]
    move_on_cleanup_by_number_rules = [item for item in move_on_cleanup_rules
        if item.BackupCountUpperLimit.ref > 0 and item.DeleteOlderBackupsTimePeriod.Value.ref == 0]
    move_on_cleanup_by_archive_size_rules = [item for item in move_on_cleanup_rules
        if item.BackupUpperLimitSize.ref > 0 and item.DeleteOlderBackupsTimePeriod.Value.ref == 0]
    source_archive, source_slices = select_slices_on_source(source_slice_selector)
    slices = []
    if move_on_cleanup_by_time_rules:
        retention_predicate = create_retention_predicate(
            move_on_cleanup_by_time_rules, backup_set_calculator, cleanup_time)
        slices = select_slices_abandoned_by_time(source_slices, retention_predicate)
    elif move_on_cleanup_by_number_rules:
        slices = select_slices_abandoned_by_number(
            source_slices, backup_set_calculator, move_on_cleanup_by_number_rules)
    elif move_on_cleanup_by_archive_size_rules:
        slices = select_slices_abandoned_by_archive_size(
            source_archive, source_slices, move_on_cleanup_by_archive_size_rules)
    if len(slices) == 0:
        return []
    target_archive, target_slices = select_slices(service, target_stage, options, ignore_missing_archive = True)
    create_archive = target_slices is None
    slice_keys = [slice['id'] for slice in slices]
    slice_uris = [slice['uri'] for slice in slices]
    requests = []
    if create_archive:
        requests = [request_for_make_shallow_copy(source_stage, target_stage, owner_id)]
    replicate_request = request_for_replicate_slices(source_stage, slice_keys, target_stage, options, owner_id)
    consolidate_backup = get_consolidate_backup_from_rules(move_on_cleanup_rules)
    cleanup_request = request_for_delete_slices(backup_plan,
        source_stage, preserve_last_slice(source_slices,slice_uris), consolidate_backup, owner_id)
    return requests + [replicate_request, cleanup_request]


def create_validation_argument(backup_plan, stage, owner_id, backup_plan_id):
    validate_last_slice_only = False
    if 'validation_options' in stage:
        validation_options = stage['validation_options']
        if not (validation_options.is_data() and validation_options.type == acrort.plain.NIL):
            validate_last_slice_only = validation_options.ValidationScope.ref == VALIDATION_SCOPE_LATEST_SLICE

    argument = {
        'PlanID': backup_plan_id,
        'ArchiveName': stage['archive_name'],
        'LocationUri': stage['location_uri'],
        'LocationCredentials': stage['location_credentials'],
        'ValidateLastSliceOnly': validate_last_slice_only,
        'OwnerID': owner_id,
    }

    if backup_plan['Options']['BackupOptions']['ArchiveProtection']['UsePassword'].ref:
        argument.update({'ArchiveCredentialsId': backup_plan['Options']['BackupOptions']['ArchiveProtection']['PasswordUrl']})
   
    return argument


def create_validation_command_request(backup_plan, source, owner_id, backup_plan_id):
    argument = create_validation_argument(backup_plan, source, owner_id, backup_plan_id)
    return PREPARE_VALIDATION_COMMAND_ID, argument


def create_archive_validation_command_request(backup_plan, source, owner_id, backup_plan_id):
    argument = create_validation_argument(backup_plan, source, owner_id, backup_plan_id)
    argument['ValidateLastSliceOnly'] = False
    return PREPARE_VALIDATION_COMMAND_ID, argument


def is_astorage_location(location_uri):
    parsed_url = urlparse(location_uri)
    return 'storage' in parse_qs(parsed_url.query)


def is_max_index(index, count):
    return index + 1 >= count


def is_route_with_initial_seeding(stages):
    route_length = len(stages)
    online_stage_index = route_length - 2
    location_uri = stages[online_stage_index]['location_uri'].ref
    return route_length > 1 and \
           stages[online_stage_index]['destination_kind'].ref == _LOCATION_KIND_ONLINE and \
           not is_astorage_location(location_uri) and \
           not is_ostor_uri(location_uri)


def create_mark_as_seeded_command_request(source, owner_id):
    argument = {
        'ArchiveName': source['archive_name'],
        'LocationUri': source['location_uri'],
        'OwnerID': owner_id,
    }
    return (MARK_AS_SEEDED_COMMAND_ID, argument)


class PlanInterface:
    def replicate(self, copy_on_appearance_rules, stage_index, stages) -> list:
        """process_replication"""
        pass

    def get_target_stage(self, stage_index, stages) -> any:
        """get_target_stage"""
        pass

    def is_location_creation_prohibited(self) -> bool:
        """is_prohibit_location_creation"""
        pass

    def is_target_pds_archive(self) -> bool:
        """is_target_pds_archive"""
        pass

    def is_last_stage(self, stage_index, stages) -> bool:
        """is_last_stage"""
        pass

    def validate(self, stage_index, stages) -> list:
        """validate"""
        pass


class DefaultPlan(PlanInterface):
    def __init__(self, service, backup_plan):
        self.service = service
        self.backup_plan = backup_plan

    def get_owner_id(self):
        return self.backup_plan['OwnerID']

    def get_options(self):
        return self.backup_plan['Options']

    def get_target_stage(self, stage_index, stages):
        return stages[stage_index + 1]

    def is_location_creation_prohibited(self) -> bool:
        return False

    def is_target_pds_archive(self) -> bool:
        return False

    def replicate(self, copy_on_appearance_rules, stage_index, stages):
        source_stage = stages[stage_index]
        source_slice_selector = lambda: select_slices(self.service, source_stage, self.get_options(), ignore_missing_archive=False)
        target_stage = self.get_target_stage(stage_index, stages)
        target_archive_is_pds = self.is_target_pds_archive()
        target_uri = target_stage['location_uri'].ref
        include_immutable = is_cloud_uri(target_uri) and not target_archive_is_pds
        ignore_missing_archive = True
        target_slice_selector = lambda: select_slices(self.service, target_stage, self.get_options(), ignore_missing_archive, include_immutable)

        prohibit_location_creation = self.is_location_creation_prohibited()

        copy_workflow = calculate_copy_on_appearance_workflow(
            copy_on_appearance_rules, self.create_backupset_calculator(), source_slice_selector, target_slice_selector,
            source_stage, target_stage, self.get_options(), self.get_owner_id(), target_archive_is_pds, prohibit_location_creation)
        return copy_workflow

    def get_rules_by_run_type(self, stage_index, stages, run_type):
        source_stage = stages[stage_index]
        is_applicable = lambda rule: rule.BeforeBackup.ref if run_type == STAGING_RUN_TYPE_BEFORE_BACKUP else rule.AfterBackup.ref
        return [rule for _, rule in source_stage['rules'] if is_applicable(rule)]

    def filter_by_operation_type(self, staging_rules, operation_type):
        return [rule for rule in staging_rules if operation_type == rule.StagingOperationType.ref]

    def get_replication_rules(self, stage_index, stages, run_type):
        staging_rules = self.get_rules_by_run_type(stage_index, stages, run_type)
        return self.filter_by_operation_type(staging_rules, STAGING_TYPE_COPY_ON_APPEARANCE)

    def get_cleanup_rules(self, stage_index, stages, run_type):
        staging_rules = self.get_rules_by_run_type(stage_index, stages, run_type)
        return self.filter_by_operation_type(staging_rules, STAGING_TYPE_JUST_CLEANUP)

    def create_backupset_calculator(self):
        schedule_type = get_schedule_type(self.backup_plan)
        week_start = get_week_beginning(self.backup_plan)
        return create_backup_set_calculator(schedule_type, week_start)

    def cleanup(self, cleanup_rules, stage_index, stages):
        cleanup_time = datetime.datetime.now(datetime.timezone.utc).astimezone()
        source_stage = stages[stage_index]
        source_slice_selector = lambda: select_slices(self.service, source_stage, self.get_options(), ignore_missing_archive=False)
        cleanup_workflow = calculate_just_cleanup_workflow(self.backup_plan,
            cleanup_rules, self.create_backupset_calculator(), cleanup_time, source_slice_selector, source_stage,
            create_retention_predicate, self.get_owner_id())
        return cleanup_workflow

    def validate(self, stage_index, stages):
        if stage_index == 0:
            # It's the backup stage. Do not schedule validation because backup command validates itself.
            return []

        # It's the replication stage. Let's schedule validation if required.
        source_stage = stages[stage_index]
        validation_options = source_stage['validation_options']
        if not (validation_options.is_data() and validation_options.type == acrort.plain.NIL):
            if validation_options.AfterFullBackup.ref or validation_options.AfterIncrementalBackup.ref or validation_options.AfterDifferentialBackup.ref:
                return [create_validation_command_request(self.backup_plan, source_stage, self.get_owner_id(), self.backup_plan["ID"])]
        return []

    def is_last_stage(self, stage_index, stages):
        return is_max_index(stage_index, len(stages))


class PdsPlan(DefaultPlan):
    def __init__(self, service, backup_plan, archive, slices):
        super().__init__(service, backup_plan)
        self.pds_archive = archive
        self.pds_slices = slices

    def is_pdsi_workflow(self):
        # pdsi workflow will be finished when pds archive will be created and validated.
        return not self.pds_archive or self.is_pdsi_archive()

    def is_pds_workflow(self):
        # pds workflow will be finished when pds archive will be delivered to online stage, that means will have slices.
        return not self.is_pdsi_workflow() and self.is_pds_archive(self.pds_archive) and (not self.pds_slices or len(self.pds_slices) == 0)

    def is_pds_workflow_completed(self):
        return not self.is_pdsi_workflow() and self.is_pds_archive(self.pds_archive) and self.pds_slices and len(self.pds_slices) > 0

    def is_last_stage(self, stage_index, stages):
        # if pds_workflow we should forbid any replication to online stage, else this is the last stage and nothing to replicate.
        return super().is_last_stage(stage_index, stages) or self.is_pds_workflow()

    def is_pds_archive(self, pds_archive):
        return pds_archive and 'pds' in pds_archive and pds_archive['pds'] != 0

    def is_pdsi_archive(self):
        return self.pds_archive and 'pdsi' in self.pds_archive and self.pds_archive['pdsi'] != 0

    def get_target_stage(self, stage_index, stages):
        if not self.is_pdsi_workflow():
            return super().get_target_stage(stage_index, stages)
        return stages[stage_index + 2]  # in case of pdsi workflow the next stage is media

    def validate(self, stage_index, stages):
        if self.is_pds_workflow_completed():
            return super().validate(stage_index, stages)
        
        return []

    def is_location_creation_prohibited(self) -> bool:
        return not self.is_pdsi_workflow()

    def is_target_pds_archive(self) -> bool:
        return True

    def replicate(self, copy_on_appearance_rules, stage_index, stages):
        copy_workflow = super().replicate(copy_on_appearance_rules, stage_index, stages)
        if not self.is_pdsi_workflow():
            return copy_workflow

        # 1. Make a pdsi archive in the online stage if it is not created yet;
        # 2. Replicate slices to the media stage;
        # 3. Validate the archive on the media stage;
        # 4. Convert the pdsi archive to pds.
        source_stage = stages[stage_index]
        online_stage_in_pds = stages[stage_index + 1]
        media_stage_in_pds = stages[stage_index + 2]
        if not self.is_pdsi_archive():
            create_pdsi_archive = True
            copy_workflow = [request_for_make_shallow_copy(source_stage, online_stage_in_pds, self.get_owner_id(), create_pdsi_archive)] + copy_workflow  # 1. & 2.
        copy_workflow += [create_archive_validation_command_request(self.backup_plan, media_stage_in_pds, self.get_owner_id(), self.backup_plan["ID"])]  # 3.
        copy_workflow += [create_mark_as_seeded_command_request(online_stage_in_pds, self.get_owner_id())]  # 4.
        return copy_workflow


def calculate_workflow(service, backup_plan, stage_index, run_type):
    if run_type not in [STAGING_RUN_TYPE_BEFORE_BACKUP, STAGING_RUN_TYPE_AFTER_BACKUP, STAGING_RUN_TYPE_ON_SCHEDULE]:
        raise ValueError('RunType argument is invalid')

    stages = [make_stage_spec(backup_plan, stage) for _, stage in backup_plan.Route.Stages]
    stage_count = len(stages)
    if stage_index >= stage_count:
        acrort.common.make_logic_error('stage_index={} while stage_count={}'.format(stage_index, stage_count)).throw()
    
    is_pds_route = is_route_with_initial_seeding(stages)

    plan = DefaultPlan(service, backup_plan)
    current_stage_is_media_in_pds = is_pds_route and plan.is_last_stage(stage_index, stages)
    if current_stage_is_media_in_pds:
        return []

    online_stage_index_in_pds = stage_count - 2
    next_stage_is_online_in_pds = is_pds_route and stage_index + 1 >= online_stage_index_in_pds
    if next_stage_is_online_in_pds:
        online_stage_in_pds = stages[online_stage_index_in_pds]
        options = backup_plan['Options']
        pds_archive, pds_slices = select_slices(service, online_stage_in_pds, options, ignore_missing_archive=True)
        plan = PdsPlan(service, backup_plan, pds_archive, pds_slices)

    # Schedule cleanup on current stage
    cleanup_rules = plan.get_cleanup_rules(stage_index, stages, run_type)
    if cleanup_rules:
        cleanup_workflow = plan.cleanup(cleanup_rules, stage_index, stages)
    else:
        cleanup_workflow = []

    pause_cdp_workflow = []
    resume_cdp_workflow = []

    if stage_index == 0 and platform.system() == 'Windows' and is_cdp_plan(backup_plan):
        pause_cdp_workflow = request_for_pause_cdp(backup_plan)
        resume_cdp_workflow = request_for_resume_cdp(backup_plan)

    if run_type == STAGING_RUN_TYPE_BEFORE_BACKUP:
        return pause_cdp_workflow + cleanup_workflow

    # Schedule validation on current stage
    validate_workflow = plan.validate(stage_index, stages)

    # It is the last stage: nowhere to replicate or replication isn't allowed
    if plan.is_last_stage(stage_index, stages):
        workflow = []
        workflow += cleanup_workflow 
        workflow += resume_cdp_workflow
        workflow += validate_workflow
        return workflow

    # It isn't the last stage. Let's calculate what we need to replicate from source to target stage if we have appropriate rules.
    replication_rules = plan.get_replication_rules(stage_index, stages, run_type)
    if replication_rules:
        copy_workflow = plan.replicate(replication_rules, stage_index, stages)
    else:
        copy_workflow = []

    workflow = []

    if len(cleanup_workflow) == 0 or len(copy_workflow) != 0:
        workflow += resume_cdp_workflow

    workflow += copy_workflow

    if len(cleanup_workflow) != 0:
        workflow += pause_cdp_workflow + cleanup_workflow + resume_cdp_workflow

    workflow += validate_workflow

    return workflow  # copy, cleanup, validate


def calculate_workflow_command(service, argument):
    backup_plan, stage_index, run_type = argument.BackupPlan, argument.StageIndex.ref, argument.RunType.ref

    try:
        workflow = calculate_workflow(service, backup_plan, stage_index, run_type)
    except SliceInFutureError as err:
        SliceInFutureErrorToLogicError(err, backup_plan.Route.Stages[stage_index]).throw()
    min_stage_progress = round(100 * stage_index / len(argument.BackupPlan.Route.Stages))
    max_stage_progress = round(100 * (stage_index + 1) / len(argument.BackupPlan.Route.Stages))
    if len(workflow):
        step_delta = round((max_stage_progress - min_stage_progress) / len(workflow))
    commands = []
    for i, (cmd, arg) in enumerate(workflow):
        min_progress = min_stage_progress + i * step_delta
        max_progress = min_stage_progress + (i + 1) * step_delta
        commands.append(make_command_spec(cmd, arg, min_progress, max_progress))
    return make_workflow(commands)
