# @copyright (c) 2002-2013 Acronis International GmbH. All rights reserved.

from .processors import HTTPBaseAuthProcessor, HTTPRedirectProcessor
from acronis.lib import json
from urllib.request import build_opener, Request
from urllib.error import HTTPError
from urllib.parse import urljoin, quote, unquote, urlsplit, urlunsplit, splitvalue, urlencode


__all__ = ['get', 'post', 'delete', 'head', 'put', 'HTTPError', 'urljoin', 'quote', 'unquote',
           'urlsplit', 'urlunsplit', 'splitvalue']


def _make_request(url, method, data=None, headers=None, form=False):
    request = Request(url, method=method, data=data)
    if isinstance(data, (dict, tuple, list)):
        if form:
            request.data = urlencode(data).encode('utf8')
            request.add_header('Content-Type', 'application/x-www-form-urlencoded; charset=utf-8')
        else:
            request.data = json.dumps(data).encode('utf8')
            request.add_header('Content-Type', 'application/json; charset=utf-8')
    elif isinstance(data, bytes):
        request.add_header('Content-Type', 'application/octet-stream')
    elif isinstance(data, str):
        request.add_header('Content-Type', 'application/text-plain; charset=utf-8')
        request.data = data.encode('utf8')
    if headers:
        for k, v in headers.items():
            request.add_header(k, v)
    return request


def _convert_response(response):
    body = response.read()
    if len(body) > 0 and response.headers.get('Content-Type', '').partition(';')[0].strip() == 'application/json':
        return json.loads(body.decode('utf8'))
    return body


def do_request(url, method='GET', data=None, processors=None, username=None, password=None, headers=None, form=False):
    if processors is None:
        processors = [HTTPRedirectProcessor()]
    if username is not None and password is not None:
        processors.append(HTTPBaseAuthProcessor(username, password))

    processors = set(processors)
    opener = build_opener(*processors)
    return _convert_response(opener.open(_make_request(url=url, method=method, data=data, headers=headers, form=form)))


def delete(url, **kwargs):
    return do_request(url, 'DELETE', **kwargs)


def get(url, **kwargs):
    return do_request(url, 'GET', **kwargs)


def head(url, **kwargs):
    return do_request(url, 'HEAD', **kwargs)


def post(url, **kwargs):
    return do_request(url, 'POST', **kwargs)


def put(url, **kwargs):
    return do_request(url, 'PUT', **kwargs)
