# @copyright (c) 2002-2013 Acronis International GmbH. All rights reserved.

from . import excepthook
from .json import dumps
from logging import *
from logging import RootLogger as _RootLogger
from logging import handlers as _handlers
import sys
import time

try:
    from .hash import ConstantHashString
except ImportError:
    ConstantHashString = str


class _LogRecordFormatted(LogRecord):
    def __str__(self):
        return 'LogRecord: %s' % self.__dict__

    def getMessage(self):
        # TODO postpone format to improve performance
        msg = str(self.msg)
        if self.args:
            msg = msg.format(*self.args)
        return msg


class JsonFormatter(Formatter):
    default_time_format = '%Y-%m-%dT%H:%M:%S'
    default_msec_format = '.%03d'

    def formatTime(self, record, datefmt=None):
        ct = self.converter(record.created)
        t = time.strftime(datefmt or self.default_time_format, ct)
        s = self.default_msec_format % record.msecs
        z = time.strftime('%z', ct) or 'Z'
        return t + s + z

    @staticmethod
    def unicode_escape_safe(value):
        if isinstance(value, bytes):
            return value.decode('unicode_escape')
        return value

    def format(self, record):
        if getattr(record, 'message', None):
            return record.message

        if record.exc_info:
            if not record.exc_text:
                record.exc_text = self.formatException(record.exc_info)

        result = {
            ConstantHashString('message'): record.getMessage(),
            ConstantHashString('@timestamp'): self.formatTime(record, self.datefmt),
            ConstantHashString('level'): record.levelname,
            ConstantHashString('process'): record.process
        }

        for k, v in record.extra.items():
            result[ConstantHashString(k)] = self.unicode_escape_safe(v)

        if record.exc_text:
            result[ConstantHashString('exception')] = record.exc_text
        if record.stack_info:
            result[ConstantHashString('stack')] = self.formatStack(record.stack_info)

        record.message = dumps(result)
        return record.message


class JsonLogger(_RootLogger):
    def __init__(self, level):
        super().__init__(level)
        import logging
        logging.root = self
        logging.Logger.root = self
        getattr(logging.Logger, 'manager', logging.Logger).root = self
        setLogRecordFactory(_LogRecordFormatted)

    def _log(self, level, msg, args, exc_info=None, **kwargs):
        sys_info = None
        try:
            fn, lno, func, sys_info = self.findCaller()
        except ValueError:
            fn, lno, func = "(unknown file)", 0, "(unknown function)"
        else:
            fn, lno, func = "(unknown file)", 0, "(unknown function)"
        if exc_info:
            if not isinstance(exc_info, tuple):
                exc_info = sys.exc_info()

        record = _LogRecordFormatted(self.name, level, fn, lno, msg, args, exc_info, func, sys_info)
        record.extra = kwargs
        self.handle(record)

    def rotate(self, *_):
        for handler in (h for h in self.handlers if isinstance(h, _RotateHandler)):
            handler.close()


class _RotateHandler(FileHandler):
    pass


class _QueueHandler(_handlers.QueueHandler):
    def __init__(self, queue):
        super().__init__(queue)
        self.lock = None

    def createLock(self):
        pass


class _QueueWriter(_handlers.QueueListener):
    def rotate(self, *_):
        for handler in (h for h in self.handlers if isinstance(h, _RotateHandler)):
            handler.close()

    def handle(self, record):
        try:
            super().handle(record)
        except Exception as e:
            print(e, file=sys.stderr)


_DEFAULT_DATE_FORMAT = '%Y-%m-%dT%H:%M:%S'
_DEFAULT_FORMAT = '[{asctime}] {levelname}: {process:d}.{pathname}:{lineno}. {message}'


def create_logger(stream=None, filename=None, level=DEBUG, fmt=_DEFAULT_FORMAT, datefmt=_DEFAULT_DATE_FORMAT, style='{', handler=None, **_):
    excepthook.setup()
    if handler is None:
        if stream is None and filename is None:
            handler = StreamHandler(sys.stderr)
        elif filename is not None:
            handler = _RotateHandler(filename)
        elif stream is not None:
            handler = StreamHandler(stream)

    formatter = JsonFormatter(fmt, datefmt, style)
    handler.setFormatter(formatter)
    handler.setLevel(level)
    root = JsonLogger(level)
    root.addHandler(handler)
    return root


def setup_log(**kwargs):
    create_logger(**kwargs)


__queue = None
__kwargs = None


def create_multi_process_logger(filename=None, **kwargs):
    from multiprocessing import Queue
    global __queue, __kwargs
    
    if __queue is None:
        __queue = Queue(-1)
        __kwargs = kwargs

    create_logger(handler=_QueueHandler(__queue), **__kwargs)
    if filename is not None:
        handler = _RotateHandler(filename, delay=True)
        handler.setFormatter(JsonFormatter())
        return _QueueWriter(__queue, handler)


def enable_debug_trace():
    setLogRecordFactory(_LogRecordFormatted)
    basicConfig(level=DEBUG, format=_DEFAULT_FORMAT, datefmt=_DEFAULT_DATE_FORMAT, stream=sys.stderr, style='{')
