# @copyright (c) 2002-2013 Acronis International GmbH. All rights reserved.

from .algorithms import for_each
from multiprocessing import cpu_count
import threading
import logging
import queue

_SHUTDOWN_REQUEST = None


class Thread(threading.Thread):
    def __init__(self, **kwargs):
        self.cancel = threading.Event()
        super().__init__(**kwargs)

    def run(self):
        self.cancel.clear()
        while not self.cancel.is_set():
            try:
                self.loop()
            except BaseException as e:
                logging.error('{0} interrupted. details {1!r}'.format(self.name, e))
                break

    def join(self, timeout=None):
        self.cancel.set()
        super().join(timeout=timeout)

    def loop(self):
        pass

    def sleep(self, ms):
        """ return true if cancel otherwise false"""
        return self.cancel.wait(ms)


class DaemonicThread(Thread):
    def __init__(self, **kwargs):
        super().__init__(**kwargs)
        self.daemon = True


class WorkerThread(Thread):
    def __init__(self, tasks, name):
        super().__init__(name=name)
        self.tasks = tasks
        self.start()

    def loop(self):
        task = self.tasks.get()
        try:
            if task is _SHUTDOWN_REQUEST:
                self.cancel.set()
            else:
                task()
        finally:
            self.tasks.task_done()


class ThreadPool:
    """Thread Pool Implementation """
    ErrorFull = queue.Full

    class ErrorDestroyed(RuntimeError):
        pass

    _threads = []

    def __init__(self, count=1, maxsize=0, timeout=None):
        self._running = True
        self.queue = queue.Queue(maxsize)
        self.queue_timeout = timeout
        self._threads = [WorkerThread(self.queue, name='Worker{}'.format(i)) for i in range(count)]

    @property
    def queue_limit(self):
        """ return current limit of associated queue"""
        return self.queue.maxsize

    @queue_limit.setter
    def queue_limit(self, maxsize):
        """ change limit of associated queue"""
        self.queue.maxsize = maxsize

    def put(self, task):
        """
            put task to thread pool
            task is a callable python object
        """
        if not self._running:
            raise ThreadPool.ErrorDestroyed('TreadPool has been already destroyed.')
        self.queue.put(task, timeout=self.queue_timeout)

    def join(self, nowait=False):
        if not self._running:
            raise ThreadPool.ErrorDestroyed('TreadPool has been already destroyed.')
        self._running = False
        while nowait:
            try:
                self.queue.get_nowait()
                self.queue.task_done()
            except queue.Empty:
                break

        for _ in self._threads:
            self.queue.put(_SHUTDOWN_REQUEST)
        if not nowait:
            self.queue.join()
            for_each(self._threads, lambda x: x.join())
        self._threads.clear()


def get_optimal_thread_count(default=4):
    cpu_number = cpu_count()
    return cpu_number if cpu_number > 1 else default
